/********************************************************************
 *This is flash ioctl for  reading ,writing and erasing application
 *******************************************************************/

#include <linux/init.h>
#include <linux/fs.h>
#include <linux/major.h>
#include <linux/module.h>
#include <linux/mtd/mtd.h>
#include <linux/cdev.h>
#include <linux/errno.h>
#include <linux/spinlock_types.h>
#include <linux/uaccess.h>
#include <linux/slab.h>
#include <linux/spi/spi.h>
#include "flash_ioctl.h"


/*
 * IOCTL Command Codes
 */
#define IPQ_FLASH_READ				0x01
#define IPQ_FLASH_WRITE				0x02
#define IPQ_FLASH_ERASE				0x03
#define IPQ_FLASH_READ_UID          0x04


#define IPQ_IO_MAGIC 				0xB3
#define	IPQ_IO_FLASH_READ			_IOR(IPQ_IO_MAGIC, IPQ_FLASH_READ, char)
#define IPQ_IO_FLASH_WRITE			_IOW(IPQ_IO_MAGIC, IPQ_FLASH_WRITE, char)
#define	IPQ_IO_FLASH_ERASE			_IO (IPQ_IO_MAGIC, IPQ_FLASH_ERASE)
#define IPQ_IO_FLASH_READ_UID       _IOR(IPQ_IO_MAGIC, IPQ_FLASH_READ_UID, char)

#define	IPQ_IOC_MAXNR				14
#define flash_major      			239
#define flash_minor      			0

#define IPQ_FLASH_SECTOR_SIZE	    (64 * 1024)
#define OK 0
#define TRUE 1
#define FALSE 0
#define FLASH_ERASE_SIZE            IPQ_FLASH_SECTOR_SIZE

/* MACRONIXPART */
#define MX_JEDEC_ID_ADDR            0X9F
#define MX_MANUFACTORY_ID           0xC2
#define MX_UID_ADDR                 0x5A

/* WINDBOND */
#define WB_JEDEC_ID_ADDR            0X90
#define WB_MANUFACTORY_ID           0xEF
#define WB_UID_ADDR                 0x4B

#define EN_FLASH_RDID               0x90
#define EN_MF_ID                    0x1C
#define EN_64_ID                    0x16
#define EN_FLASH_RDUID              0x5A    /* read unique id */
#define EN_FLASH_RDUID_ADDR         0x80
#define EN_UID_LEN                  (12)

/* XM25QH128A */
#define XM_FLASH_RDID               0x90
#define XM_MF_ID                    0x20
#define XM25QH64_ID                 0x16
#define XM_FLASH_RDUID              0x5A    /* read unique id */
#define XM_FLASH_RDUID_ADDR         0x80
#define XM_UID_LEN                  (12)

int ipq_flash_ioctl(struct file *file,  unsigned int cmd, unsigned long arg);
int ipq_flash_open (struct inode *inode, struct file *file);
extern struct mtd_info *ipq_get_mtd(void);
extern struct spi_device *ipq_get_spi(void);

struct file_operations flash_device_op = {
        .owner = THIS_MODULE,
        .unlocked_ioctl = ipq_flash_ioctl,
        .open = ipq_flash_open,
};

static struct cdev flash_device_cdev = {
        .owner  = THIS_MODULE,
		.ops = &flash_device_op,
};

typedef struct 
{
	u_int32_t addr;		/* flash r/w addr	*/
	u_int32_t len;		/* r/w length		*/
	u_int8_t* buf;		/* user-space buffer*/
	u_int32_t buflen;	/* buffer length	*/
	u_int32_t hasHead;	/* has head or not	*/
}ARG;


struct mtd_info *mtd = NULL;

static inline void 
nvram_flash_sector_erase(struct mtd_info *mtd, uint32_t addr)
{
    int ret = 0;
    struct erase_info *erase = NULL;

    erase = (struct erase_info *)kmalloc(sizeof(struct erase_info), GFP_KERNEL);
    if (!erase)
    {
        printk("Erase failed, Alloc erase(%p) failed\n", erase);
        goto end;
    }

    erase->addr = addr;
    erase->len  = FLASH_ERASE_SIZE;
    erase->mtd  = mtd;
    erase->callback = NULL;
    erase->priv = 0;
    ret = mtd_erase(mtd, erase);
    if (ret) 
    {
        printk("Erase failed, call mtd_erase failed, ret:%d\n", ret);
        goto end;
    }

end:
    if (erase) 
    { 
        kfree(erase);
    }
    return;

}

int nvram_flash_read(struct mtd_info *mtd, u_int8_t *tempData, 
    u_int32_t hasHead, u_int32_t offset, u_int8_t *data, u_int32_t len)
{
    int i = 0;
    int ret = 0;
    size_t retlen = 0;
	
    int nSector = len / (IPQ_FLASH_SECTOR_SIZE); 	/* Divide IPQ_FLASH_SECTOR_SIZE */	
    size_t oddLen  = len % (IPQ_FLASH_SECTOR_SIZE);	/* odd length (0 ~ IPQ_FLASH_SECTOR_SIZE) */

    /* copy from flash to kernel buffer */  
    for (i = 0; i < nSector; i++)
    {		
        ret = mtd_read(mtd, offset, IPQ_FLASH_SECTOR_SIZE, &retlen, tempData);
        if ((ret == 0) && (retlen == IPQ_FLASH_SECTOR_SIZE))
        {
            /*copy from kernel to user*/
            if (copy_to_user(data, tempData, IPQ_FLASH_SECTOR_SIZE))
            {
                printk("read copy_to_user failed\n");
                goto wrong;
            }
        }
        else
        {
            printk("Read failed, ret:%d!\n", ret);
            goto wrong;
        }
        offset += IPQ_FLASH_SECTOR_SIZE;
        data += IPQ_FLASH_SECTOR_SIZE;
    }
   
    if (oddLen) 
    {
        ret = mtd_read(mtd, offset, oddLen, &retlen, tempData);
        if ((ret == 0) && (retlen == oddLen))
        {
            /*copy from kernel to user*/
            if (copy_to_user(data, tempData, oddLen))
            {
                printk("read copy_to_user failed\n");
                goto wrong;
            }
        } 
        else 
        {
            printk("Read failed!\n");
            goto wrong;
        }
    }
    return 0;

wrong:
    return -1;
}

int nvram_flash_write(struct mtd_info *mtd, u_int8_t *tempData, 
    u_int32_t hasHead, u_int32_t offset, u_int8_t *data, u_int32_t len)
{
	u_int32_t address = 0;
	u_int32_t headLen = 0;
	u_int32_t endAddr = 0, startAddr = 0;
	u_int8_t *orignData = NULL;
	u_int32_t headData[2] = {len, 0};
	u_int32_t frontLen = 0, tailLen = 0;
	size_t retLen = 0;

	headData[0] = htonl(len);	

	if (hasHead != FALSE)
	{
		headLen = 2 * sizeof(u_int32_t);
		len += headLen;
	}

	frontLen = offset % FLASH_ERASE_SIZE;
	tailLen  = (offset + len) % FLASH_ERASE_SIZE;
	/* 第一个block起始地址 */
	address = offset - frontLen;
	/* 最后一个不完整的block起始地址，如果没有，则是再下一个block起始地址 */
	endAddr = offset + len - tailLen;

	orignData = tempData + FLASH_ERASE_SIZE;

	if (frontLen > 0 || headLen > 0)/* 第一个block */
	{
		mtd_read(mtd, address, FLASH_ERASE_SIZE, &retLen, orignData);
		memcpy(tempData, orignData, frontLen);/* 前面一部分为原来的的数据 */
		
		if (FLASH_ERASE_SIZE < frontLen + headLen) /* 头部被拆分到两个blcok */
		{
			headLen = FLASH_ERASE_SIZE - frontLen;
			/* 分区头部，第一部分 */
			memcpy(tempData + frontLen, headData, headLen);

			/***************************************************/
			if (memcmp(orignData, tempData, FLASH_ERASE_SIZE)) /*内容变化*/
			{
				nvram_flash_sector_erase(mtd,address);
				mtd_write(mtd, address, FLASH_ERASE_SIZE, &retLen, tempData);
			}
			address += FLASH_ERASE_SIZE;
			/***************************************************/

			mtd_read(mtd, address, FLASH_ERASE_SIZE, &retLen, orignData);
			/* 分区头部，第二部分 */
			memcpy(tempData, (u_int8_t*)(headData) + headLen, 8 - headLen);

			if (len - headLen < FLASH_ERASE_SIZE) /*写入数据长度小于一个block*/
			{
				headLen = 8 - headLen;
				copy_from_user(tempData + headLen, data, tailLen - headLen);/* 需要写入的数据 */
				memcpy(tempData + tailLen, orignData + tailLen, FLASH_ERASE_SIZE - tailLen);/* 原来的数据 */
				data += tailLen - headLen;
			}
			else
			{
				headLen = 8 - headLen;
				copy_from_user(tempData + headLen, data, FLASH_ERASE_SIZE - headLen);/* 需要写入的数据 */
				data += FLASH_ERASE_SIZE - headLen;
			}
		}
		else /* 头部未被拆分 */
		{
			memcpy(tempData + frontLen, headData, headLen);/* 分区头部(如果有的话) */
			
			if (len + frontLen < FLASH_ERASE_SIZE) /*写入数据长度小于一个block*/
			{
				copy_from_user(tempData + frontLen + headLen, data, len - headLen);/* 后面为需要写入的数据 */
				data += len - headLen;
				/* 再后面是原来的数据 */
				memcpy(tempData + frontLen + len,
						orignData + frontLen + len,
						FLASH_ERASE_SIZE - (frontLen + len));
			}
			else
			{
				copy_from_user(tempData + frontLen + headLen, data, FLASH_ERASE_SIZE - frontLen - headLen);
				/* 后面为需要写入的数据 */
				data += FLASH_ERASE_SIZE - frontLen - headLen;
			}
		}

		/***************************************************/
		if (memcmp(orignData, tempData, FLASH_ERASE_SIZE)) /*内容变化*/
		{
		    nvram_flash_sector_erase(mtd,address);
		    mtd_write(mtd, address, FLASH_ERASE_SIZE, &retLen, tempData);
		}
		address += FLASH_ERASE_SIZE;
		/***************************************************/
	}

	if (address < endAddr)/* 中间完整的block，注意:此处用 < 而不是 <=。 */
	{
		startAddr = address;
		while (address < endAddr)
		{
			mtd_read(mtd, address, FLASH_ERASE_SIZE, &retLen, orignData);
			copy_from_user(tempData, data, FLASH_ERASE_SIZE);
			/***************************************************/
			if (memcmp(orignData, tempData, FLASH_ERASE_SIZE)) /*内容变化*/
			{
				nvram_flash_sector_erase(mtd,address);
				mtd_write(mtd, address, FLASH_ERASE_SIZE, &retLen, tempData);
			}
			address += FLASH_ERASE_SIZE;
			/***************************************************/
			data += FLASH_ERASE_SIZE;
		}
	}

	if (address < offset + len) /* 如果还没有写完，则说明最后有一个不完整的block */
	{
		mtd->_read(mtd, address, FLASH_ERASE_SIZE, &retLen, orignData);
		copy_from_user(tempData, data, tailLen);/*前面一部分为需要写入的数据*/
		memcpy(tempData + tailLen, orignData + tailLen, FLASH_ERASE_SIZE - tailLen);
		/*后面为原来数据*/
		/***************************************************/
		if (memcmp(orignData, tempData, FLASH_ERASE_SIZE)) /*内容变化*/
		{
		    nvram_flash_sector_erase(mtd,address);
		    mtd_write(mtd, address, FLASH_ERASE_SIZE, &retLen, tempData);
		}
		address += FLASH_ERASE_SIZE;
		/***************************************************/
	}

	return OK;
}

int 
ipq_spi_get_jedec_id(void)
{
    int ret = 0;
    u8 code = 0;
    u8 data[5] = {0};
    u8 cmds[5] = {0};
	struct spi_device *spi = ipq_get_spi();

    if(spi == NULL){
        printk("get spi fail\n");
        return;
    }

    /* try to get xm jdedec id */
    cmds[0] = XM_FLASH_RDID;
    cmds[1] = 0;
    cmds[2] = 0;
    cmds[3] = 0;
    ret = spi_write_then_read(spi, &cmds, 4, data, 1);
    if (ret < 0){
        printk("read wb jdedec fail, ret=%d\n", ret);
        return -1;
    }

    if(data[0] == XM_MF_ID){
        return XM_MF_ID;
    }

    /* try to get en jdedec id */
    cmds[0] = EN_FLASH_RDID;
    cmds[1] = 0;
    cmds[2] = 0;
    cmds[3] = 0x0;
    ret = spi_write_then_read(spi, &cmds, 4, data, 1);
    if (ret < 0){
        printk("read wb jdedec fail, ret=%d\n", ret);
        return -1;
    }
    if(data[0] == EN_MF_ID){
        return EN_MF_ID;
    }


    /* try to get mx jdedec id */
    code = MX_JEDEC_ID_ADDR;
	ret = spi_write_then_read(spi, &code, 1, data, 1);
	if (ret < 0){
		printk("read mx jdedec fail, ret=%d\n", ret);
        return -1;
	}
    if(data[0] == MX_MANUFACTORY_ID){
        return MX_MANUFACTORY_ID;
    }

    /* try to get wb jdedec id */
    code = WB_JEDEC_ID_ADDR;
	ret = spi_write_then_read(spi, &code, 1, data, 5);
	if (ret < 0){
		printk("read wb jdedec fail, ret=%d\n", ret);
        return -1;
	}
    if(data[3] == WB_MANUFACTORY_ID){
        return WB_MANUFACTORY_ID;
    }

    return -1;
}

int
ipq_spi_uid_read(u8 *buf, int length)
{
    int ret = 0;
    int jedec_id = 0;
    u8  code[5];
    u8  out[32];
	struct spi_device *spi = ipq_get_spi();

    if(spi == NULL){
        printk("get spi fail\n");
        return -1;
    }

    jedec_id = ipq_spi_get_jedec_id();
    switch(jedec_id)
    {
    case MX_MANUFACTORY_ID:
        if (length < 16)
        {
            printk("mx flash uid len is 16 bytes, buf length to short!!!\n");
            return -1;
        }

        /* UID ADDRESS 24-bit begin from 0x8C */
        code[0] = MX_UID_ADDR;
        code[1] = 0;
        code[2] = 0;
        code[3] = 0x8C;

        memset(out, 0, 17);
        ret = spi_write_then_read(spi, code, 4, out, 14);
        if (ret < 0){
            printk("read wb uid fail, ret=%d\n", ret);
            return -1;
        }

        /* out[0] is dummy, out[1]~out[13] is uid */
        memcpy(buf, &out[1], 13);
        return 16;

    case WB_MANUFACTORY_ID:
        if (length < 8)
        {
            printk("wb flash uid len is 8 bytes, buf length to short!!!\n");
            return -1;
        }

        code[0] = WB_UID_ADDR;
        memset(out, 0, 16);
        ret = spi_write_then_read(spi, code, 1, out, 16);
        if (ret < 0){
            printk("read wb uid fail, ret=%d\n", ret);
            return -1;
        }

        /* out[0]~out[4] is dummy, out[5]~out[12] is uid */
        memcpy(buf, &out[5], 8);
        return 8;

    case XM_MF_ID:
        if (length < XM_UID_LEN)
        {
            printk("xm flash uid len is %d bytes, buf length to short!!!\n", XM_UID_LEN);
            return -1;
        }

        code[0] = XM_FLASH_RDUID;
        code[1] = 0;
        code[2] = 0;
        code[3] = XM_FLASH_RDUID_ADDR;
        memset(out, 0, 16);
        ret = spi_write_then_read(spi, code, 4, out, XM_UID_LEN+1);
        if (ret < 0){
            printk("read wb uid fail, ret=%d\n", ret);
            return -1;
        }

        /* out[0] is dummy, out[1]~out[12] is uid */
        memcpy(buf, &out[1], XM_UID_LEN);
        return XM_UID_LEN;

    case EN_MF_ID:
        if (length < EN_UID_LEN)
        {
            printk("en flash uid len is %d bytes, buf length to short!!!\n", EN_UID_LEN);
            return -1;
        }

        code[0] = EN_FLASH_RDUID;
        code[1] = 0;
        code[2] = 0;
        code[3] = EN_FLASH_RDUID_ADDR;
        memset(out, 0, 16);
        ret = spi_write_then_read(spi, code, 4, out, EN_UID_LEN + 1);
        if (ret < 0){
            printk("read wb uid fail, ret=%d\n", ret);
            return -1;
        }

        /* out[0] is dummy, out[1]~out[12] is uid */
        memcpy(buf, &out[1], EN_UID_LEN);
        return EN_UID_LEN;

    default:
        printk("unknown flash \n");
        return -1;
    }
}

int ipq_flash_ioctl(struct file *file,  unsigned int cmd, unsigned long arg)
{
	/* temp buffer for r/w */
	unsigned char *rwBuf = NULL;
	
	ARG *pArg = (ARG*)arg;
	u_int8_t* usrBuf = pArg->buf;
	u_int32_t usrBufLen = pArg->buflen;
	u_int32_t addr = pArg->addr;
	u_int32_t hasHead = pArg->hasHead;
	int ret = 0;
	
	rwBuf = (unsigned char *)kmalloc(2*FLASH_ERASE_SIZE, GFP_KERNEL);
	if (NULL == rwBuf)
	{
		printk("%s: failed to kmalloc!\n", __func__);
		return -1;
	}

	if (_IOC_TYPE(cmd) != IPQ_IO_MAGIC) {
		printk("cmd type error!\n");
		goto wrong;
	}
	if (_IOC_NR(cmd) > IPQ_IOC_MAXNR) {
		printk("cmd NR error!\n");
		goto wrong;
	}
    
	if (_IOC_DIR(cmd) & _IOC_READ) {
		ret = access_ok(VERIFY_WRITE, (void __user *)usrBuf, _IOC_SIZE(cmd));
	} else if (_IOC_DIR(cmd) & _IOC_WRITE) {
		ret = access_ok(VERIFY_READ, (void __user *)usrBuf, _IOC_SIZE(cmd));
	}
    
	if (ret < 0) { 
		printk("access %p not ok!\n", usrBuf);
		goto wrong;
	}
	
	switch(cmd)
	{
	case IPQ_IO_FLASH_READ:
		ret = nvram_flash_read(mtd, rwBuf, hasHead, addr, usrBuf, usrBufLen);
		break;

	case IPQ_IO_FLASH_WRITE:
		ret = nvram_flash_write(mtd, rwBuf, hasHead, addr, usrBuf, usrBufLen);
		break;

	case  IPQ_FLASH_ERASE:
		ret = -1;
		break;
		
	case IPQ_IO_FLASH_READ_UID:
		ret = ipq_spi_uid_read(usrBuf, usrBufLen);
		break;
	}

wrong:

	if (rwBuf) {
		kfree(rwBuf);
	}

	return ret;
}


int ipq_flash_open (struct inode *inode, struct file *filp)
{
	int minor = iminor(inode);

	if (mtd == NULL)
	{
		mtd = ipq_get_mtd();
		if ( mtd == NULL )
		{
			printk(" ipq_get_mtd Can not open mtd!\n");
			return 0;
		}
	}

	if ((filp->f_mode & 2) && (minor & 1)) {
		printk("You can't open the RO devices RW!\n");
		return -EACCES;
	}
	
	return 0;
	
}

#ifdef CONFIG_TPLINK_MEM_ROOTFS
static struct mtd_info *mtdram_info = NULL;
extern int del_mtd_device(struct mtd_info *mtd);
extern struct mtd_info *get_mtd_device_nm(const char *name);
extern int add_mtd_device(struct mtd_info *mtd);

static int mtdram_read(struct mtd_info *mtd, loff_t from, size_t len,
		size_t *retlen, u_char *buf)
{
	if (from + len > mtd->size)
		return -EINVAL;

	memcpy(buf, mtd->priv + from, len);

	*retlen = len;
	return 0;
}

static void mtdram_cleanup(void)
{
    if (mtdram_info) {
        del_mtd_device(mtdram_info);
        kfree(mtdram_info);
        mtdram_info = NULL;
    }
}

static int mtdram_setup(void)
{
    struct mtd_info *mtd;
    extern unsigned long mtd_ram_addr;
    extern unsigned long mtd_ram_size;

    if (mtd_ram_addr == 0 || mtd_ram_size == 0) {
        return 0;
    }
    printk(KERN_NOTICE "%s: booting with mem rootfs@%x/%x.\n",
        __func__, mtd_ram_addr, mtd_ram_size);

    mtd = get_mtd_device_nm("rootfs");
    if (mtd != NULL && mtd != ERR_PTR(-ENODEV)) {
        put_mtd_device(mtd);
        del_mtd_device(mtd);
    } else {
        return -ENODEV;
    }
    
	mtd = kmalloc(sizeof(struct mtd_info), GFP_KERNEL);
	memset(mtd, 0, sizeof(*mtd));

	mtd->name = "rootfs";
	mtd->type = MTD_ROM;
	mtd->flags = MTD_CAP_ROM;
	mtd->size = mtd_ram_size;
	mtd->writesize = 1;
	mtd->priv = (void*)mtd_ram_addr;

	mtd->owner = THIS_MODULE;
	mtd->_read = mtdram_read;

    mtdram_info = mtd;
	if (add_mtd_device(mtd)) {
		return -EIO;
	}

	return 0;
}
#endif

int __init ipq_flash_chrdev_init (void)
{
    dev_t dev;
    int result;
    int err;
    int ipq_flash_major = flash_major;
    int ipq_flash_minor = flash_minor;

    if (ipq_flash_major) {
        dev = MKDEV(ipq_flash_major, ipq_flash_minor);
        result = register_chrdev_region(dev, 1, "flash_chrdev");
    } else {
        result = alloc_chrdev_region(&dev, ipq_flash_minor, 1, "flash_chrdev");
        ipq_flash_major = MAJOR(dev);
    }

    if (result < 0) {
        printk(KERN_WARNING "flash_chrdev : can`t get major %d\n", ipq_flash_major);
        return result;
    }
    cdev_init (&flash_device_cdev, &flash_device_op);
    err = cdev_add(&flash_device_cdev, dev, 1);
    if (err) printk(KERN_NOTICE "Error %d adding flash_chrdev ", err);

	printk("%s() %d:\t register flash_chrdev ok\n", __FUNCTION__, __LINE__);
	
#ifdef CONFIG_TPLINK_MEM_ROOTFS
    mtdram_setup();
#endif

	mtd = ipq_get_mtd();
	if ( mtd == NULL )
	{
		printk("ipq_get_mtd Can not open mtd!\n");
		return 0;
	}

    return 0;

}

static void __exit cleanup_ipq_flash_chrdev_exit (void)
{
//	unregister_chrdev_region(MKDEV(flash_major, flash_minor), 1);
#ifdef CONFIG_TPLINK_MEM_ROOTFS
    mtdram_cleanup();
#endif
}


module_init(ipq_flash_chrdev_init);
module_exit(cleanup_ipq_flash_chrdev_exit);
MODULE_LICENSE("GPL");

